/*:
 * @plugindesc モンプチ・なべミニゲーム
 * @author 夕多丸
 * 
 * @help
 * ■ プラグインコマンド
 * 
 * -----------------------------------------------------
 * NABE_SET switch file type good bad
 * -----------------------------------------------------
 * 鍋の基本設定を用意
 * =====================================================
 * switch  : 判定スイッチ番号
 * file    : ファイル名
 * type    : 演出タイプ
 * good    : 良アイテムリスト
 * bad     : 悪アイテムリスト
 * =====================================================
 * 
 * -----------------------------------------------------
 * NABE_AREA x y width height
 * -----------------------------------------------------
 * 鍋の判定範囲を設定
 * =====================================================
 * x      : 鍋の判定位置 X
 * y      : 鍋の判定位置 Y
 * width  : 鍋の判定サイズ 横幅
 * height : 鍋の判定サイズ 縦幅
 * 
 * -----------------------------------------------------
 * NABE_START
 * -----------------------------------------------------
 * 鍋のミニゲームを開始
 * 
/*:ja
 * @param se_grap
 * @desc つかんだ時のSE
 * @default Cursor1
 * @require 1
 * @dir audio/se/
 * @type file
 * 
 * @param se_release
 * @desc 放した時のSE
 * @default Cursor1
 * @require 1
 * @dir audio/se/
 * @type file
 * 
 * @param se_trash
 * @desc 捨てた時のSE
 * @default Cursor1
 * @require 1
 * @dir audio/se/
 * @type file
 *
 * @param se_end
 * @desc 完了時のSE
 * @default Cursor1
 * @require 1
 * @dir audio/se/
 * @type file
 *
 * @param se_reset
 * @desc リセット時のSE
 * @default Cursor1
 * @require 1
 * @dir audio/se/
 * @type file
 *
 **/
(function () {
  const parameters = PluginManager.parameters("MonpuchiNabe");
  const se = {
    grap: (parameters["se_grap"] || "Cursor1"),
    release: (parameters["se_release"] || "Cursor1"),
    trash: (parameters["se_trash"] || "Cursor1"),
    end: (parameters["se_end"] || "Cursor1"),
    reset: (parameters["se_reset"] || "Cursor1")
  };

  const _pluginCommand = Game_Interpreter.prototype.pluginCommand;
  Game_Interpreter.prototype.pluginCommand = function (command, args) {
    _pluginCommand.call(this, command, args);
    if (command.toUpperCase() === "NABE") {
      this.nabe = new Nabe(args);
      this.nabe.se = se;
    }
    if (command.toUpperCase() === "NABE_SET") {
      this.nabe.setOption(args);
    }
    if (command.toUpperCase() === "NABE_BUTTON") {
      this.nabe.setButton(args);
    }
    if (command.toUpperCase() === "NABE_START") {
      this.nabe.init();
      this.setWaitMode("nabe");
    }
  };

  const _updateWaitMode = Game_Interpreter.prototype.updateWaitMode;
  Game_Interpreter.prototype.updateWaitMode = function () {
    if (this._waitMode === "nabe") {
      const active = this.nabe.update();
      if (active === false) {
        this.nabe = null;
        this._waitMode = "";
      }
      return active;
    }
    return _updateWaitMode.call(this);
  };
})();

class Nabe {
  // NABE_SET 101 nabe_katatumuri/item 0 [1,2,3,4,5,6] [7,8,9,10]
  constructor(args) {
    this.switchNo = args[0];
    this.file = args[1];
    this.speed = (args[2] === 0) ? 1 / 2 : args[2] / 2;
    this.items = this.getItems(args[3], args[4]);
    this.button = {
      filename: "nabe_kansei",
      x: 560,
      y: 60,
      sprite: null
    };
    //
    this.area = [110, 110, 500, 525];
    this.isEnd = false;
    this.isPressed = false;
    this.loading = true;
    this.isKeyboard = false;
    //
    this.se = {};
    this.lastTime = new Date().getTime();
    this.mouse = { tx: TouchInput.x, ty: TouchInput.y };
    this.cursor = { x: 0, y: 0 };
  }

  init() {
    this.createImages();
    this.createButtons();
    this.createCursor();
  }

  setOption(args) {
    switch (args[0]) {
      case "area":
        this.area = [args[1], args[2], args[3], args[4]];
    }
  }

  setButton(args) {
    this.button.filename = args[0];
    this.button.x = args[1];
    this.button.y = args[2];
  }

  getItems(good, bad) {
    let items = [];
    const addItem = (vn, vbad) => {
      items.push({
        index: vn,
        isBat: vbad,
        isGrap: false,
        ready: false,
        count: 0,
        toX: 0,
        toY: 0
      });
    }
    let vgood = JSON.parse(good);
    while (vgood.length > 0) addItem(vgood.pop(), false);
    let vbad = JSON.parse(bad);
    while (vbad.length > 0) addItem(vbad.pop(), true);
    return items;
  }

  createCursor() {
    const filename = "cursor";
    const bitmap = ImageManager.loadSystem(filename);
    const sprite = new Sprite(bitmap);
    this.cursor.sprite = sprite;
    SceneManager._scene.addChild(sprite);
  }

  createImages() {
    this.items.forEach((item) => {
      const filename = this.file + "/item" + item.index;
      const bitmap = ImageManager.loadPicture(filename);
      const sprite = new Sprite(bitmap);
      item.sprite = sprite;
    });
    this.items.reverse();
  }

  createButtons() {
    const bitmap = ImageManager.loadPicture(this.button.filename);
    const sprite = new Sprite(bitmap);
    sprite.x = this.button.x;
    sprite.y = this.button.y;
    SceneManager._scene.addChild(sprite);
    this.button.sprite = sprite;
    this.button.func = () => {
      this.isEnd = true;
    }
  }

  update() {
    if (this.lastTime > new Date().getTime() - 5) return true;
    this.lastTime = new Date().getTime();
    //
    if (this.isEnd === true) return this.updateEnd();
    if (this.loading === true) return this.updateLoading();
    //
    this.updateMove();
    this.updateCursor();
    this.updateMouse();
    this.updateTargetItem();
    if (this.isKeyboard === true) {
      // キー操作の場合
      if (Input.isPressed("ok") === true) {
        if (this.isPressed === false) this.setGrapItems();
        this.moveGrapItems();
      } else {
        if (this.isPressed === true) this.setGrapEnd();
      }
    } else {
      if (TouchInput.isPressed() === true) {
        if (this.isPressed === false) this.setGrapItems();
        this.moveGrapItems();
      } else {
        if (this.isPressed === true) this.setGrapEnd();
      };
    }
    //
    return true;
  }

  updateMouse() {
    if (this.mouse.tx !== TouchInput.x && this.mouse.ty !== TouchInput.y) {
      this.mouse.tx = TouchInput.x;
      this.mouse.ty = TouchInput.y;
      this.isKeyboard = false;
      SceneManager._scene.removeChild(this.cursor.sprite);
    }
  }
  updateCursor() {
    let vx = 0;
    let vy = 0;
    if (Input.isPressed("up") === true) vy -= 7;
    if (Input.isPressed("down") === true) vy += 7;
    if (Input.isPressed("left") === true) vx -= 7;
    if (Input.isPressed("right") === true) vx += 7;
    if (vx !== 0 || vy !== 0) { // キーが入力された
      const vdia = (vx !== 0 && vy !== 0 && Math.abs(vx) + Math.abs(vy) > 5);
      this.cursor.x += (vdia === false) ? vx : Math.floor(vx * 0.8);
      this.cursor.y += (vdia === false) ? vy : Math.floor(vy * 0.8);
      if (this.cursor.x < 0) this.cursor.x = 0;
      if (this.cursor.y < 0) this.cursor.y = 0;
      if (this.cursor.x > 816) this.cursor.x = 816;
      if (this.cursor.y > 624) this.cursor.y = 624;
      //
      if (this.isKeyboard !== true) { // キー操作開始
        this.isKeyboard = true;
        SceneManager._scene.addChild(this.cursor.sprite);
      }
      this.cursor.sprite.x = this.cursor.x;
      this.cursor.sprite.y = this.cursor.y;
    }
  }

  updateLoading() {
    let count = 0;
    const vw = this.area[2] - this.area[0];
    const vh = this.area[3] - this.area[1];
    this.items.forEach((item) => {
      if (item.ready === true) return;
      if (item.sprite.width !== 0 && item.sprite.height !== 0) {
        SceneManager._scene.addChild(item.sprite);
        item.sprite.x = this.area[0] + Math.floor(Math.random() * (vw - item.sprite.width));
        item.sprite.y = this.area[1] + Math.floor(Math.random() * (vh - item.sprite.height));
        item.toX = (Math.floor(Math.random() * 5) + 1) / 10;
        item.toY = (Math.floor(Math.random() * 5) + 1) / 10;
        if (Math.floor(Math.random() * 10) >= 5) item.toX *= -1;
        if (Math.floor(Math.random() * 10) >= 5) item.toY *= -1;
        item.ready = true;
        item.isDiscarded = false;
      }
      count++;
    });
    if (count === 0) this.loading = false;
    return true;
  }

  updateMove() {
    this.items.forEach((item, i) => {
      if (item.isGrap === true) {
        item.count += 10;
      } else {
        const limit = [
          this.area[0] - item.sprite.x,
          item.sprite.x + item.sprite.width - this.area[2],
          this.area[1] - item.sprite.y,
          item.sprite.y + item.sprite.height - this.area[3]
        ];
        limit.forEach((v, i) => limit[i] = (v < 0) ? 0 : v);
        //
        if (item.toX > 0) {
          if (limit[1] > 0) item.toX *= -1;
        } else {
          if (limit[0] > 0) item.toX *= -1;
        }
        if (item.toY > 0) {
          if (limit[3] > 0) item.toY *= -1;
        } else {
          if (limit[2] > 0) item.toY *= -1;
        }
        const xg = (limit[0] + limit[1] > 0) ? 1 + Math.floor(limit[0] + limit[1]) / 4 : 1;
        const yg = (limit[2] + limit[3] > 0) ? 1 + Math.floor(limit[2] + limit[3]) / 4 : 1;
        item.sprite.x += item.toX * xg * (this.speed);
        item.sprite.y += item.toY * yg * (this.speed);
        item.count++;
      }
      item.sprite.rotation = Math.sin(item.count / 50) / 5;
    });
  }

  updateTargetItem() {
    const vx = (this.isKeyboard === false) ? TouchInput.x : this.cursor.x;
    const vy = (this.isKeyboard === false) ? TouchInput.y : this.cursor.y;
    //
    const areaCheck = (spr) => {
      if (vx < Number(spr.x)) return false;
      if (vy < Number(spr.y)) return false;
      if (vx > Number(spr.x) + Number(spr.width)) return false;
      if (vy > Number(spr.y) + Number(spr.height)) return false;
      return true;
    }
    let find = false;
    this.items.reverse();
    this.items.forEach((item) => {
      if (this.isPressed === true) {
        item.sprite.filters = [];
      } else {
        if (find === false) {
          const hit = (item.isDiscarded === true) ? false : areaCheck(item.sprite);
          if (hit === true) {
            find = true;
            item.sprite.filters = [new PIXI.filters.GlowFilter()];
          } else {
            item.sprite.filters = [];
          }
        } else {
          item.sprite.filters = [];
        }  
      }
    });
    this.items.reverse();
  }

  setGrapItems() {
    this.isPressed = true;
    //
    const vx = (this.isKeyboard === false) ? TouchInput.x : this.cursor.x;
    const vy = (this.isKeyboard === false) ? TouchInput.y : this.cursor.y;
    const areaCheck = (spr) => {
      if (vx < Number(spr.x)) return false;
      if (vy < Number(spr.y)) return false;
      if (vx > Number(spr.x) + Number(spr.width)) return false;
      if (vy > Number(spr.y) + Number(spr.height)) return false;
      return true;
    }
    let find = false;
    this.items.reverse();
    this.items.forEach((item) => {
      if (find === false) {
        item.isGrap = (item.isDiscarded === true) ? false : areaCheck(item.sprite);
        if (item.isGrap === true) {
          item.sprite.pivot.x = vx - item.sprite.x;
          item.sprite.pivot.y = vy - item.sprite.y;
          find = true;
          //PLAY-SE
          AudioManager.playStaticSe({
            name: this.se.grap,
            pitch: 100,
            volume: 100
          });
        }
      } else {
        item.isGrap = false;
      }
    });
    this.items.reverse();
    if (find === true) return;
    if (areaCheck(this.button.sprite)) this.button.func();
  }

  isDiscardArea(spr) {
    if (spr.y + spr.height - spr.pivot.y < this.area[0]) return true;
    if (spr.x + spr.width - spr.pivot.x < this.area[1]) return true;
    if (spr.x - spr.pivot.x > this.area[2]) return true;
    if (spr.y - spr.pivot.y > this.area[3]) return true;
    return false;
  }

  isNabeArea(spr) {
    if (spr.y - spr.width < this.area[0]) return true;
    if (spr.x - spr.height < this.area[1]) return true;
    if (spr.x > this.area[2]) return true;
    if (spr.y > this.area[3]) return true;
    return false;
  }

  setGrapEnd() {
    this.isPressed = false;
    this.items.filter(q => q.isGrap === true).forEach((item) => {
      item.isGrap = false;
      item.sprite.x -= item.sprite.pivot.x;
      item.sprite.y -= item.sprite.pivot.y;
      item.sprite.pivot.x = 0;
      item.sprite.pivot.y = 0;
      item.isDiscarded = this.isDiscardArea(item.sprite);
      if (item.isDiscarded === true) {
        item.sprite.alpha = 0;
        item.isDiscarded = true;
      }
      //PLAY-SE
      AudioManager.playStaticSe({
        name: (item.isDiscarded) ? this.se.trash : this.se.release,
        pitch: 100,
        volume: 100
      });
    });
  }

  moveGrapItems() {
    const grappingItems = this.items.filter(q => q.isGrap === true)
    grappingItems.forEach((item) => {
      item.sprite.x = (this.isKeyboard === false) ? TouchInput.x : this.cursor.x;
      item.sprite.y = (this.isKeyboard === false) ? TouchInput.y : this.cursor.y;
      if (this.isDiscardArea(item.sprite) === true) {
        item.sprite.alpha = 0.25;
      } else {
        item.sprite.alpha = 1;
      }
    })
  }

  shuffle() {
    if (this.pattern !== undefined) return this.playPattern();
    let result = false;
    const sel = Math.floor(Math.random() * 4);
    if (sel === 0) result = this.movePieceY(1);
    if (sel === 1) result = this.movePieceY(-1);
    if (sel === 2) result = this.movePieceX(-1);
    if (sel === 3) result = this.movePieceX(1);
    if (result === true) {
      if (this.shuffleLast === 0 && sel === 1) result = false;
      if (this.shuffleLast === 1 && sel === 0) result = false;
      if (this.shuffleLast === 2 && sel === 3) result = false;
      if (this.shuffleLast === 3 && sel === 2) result = false;
      if (result === true) this.log = this.log + String(sel);
    }
    return result;
  }

  playPattern() {
    let result = false;
    if (this.pattern.length === 0) return;
    const sel = Number(this.pattern.substr(0, 1));
    this.pattern = this.pattern.substr(1, this.pattern.length - 1);
    if (sel === 0) result = this.movePieceY(1);
    if (sel === 1) result = this.movePieceY(-1);
    if (sel === 2) result = this.movePieceX(-1);
    if (sel === 3) result = this.movePieceX(1);
    return result;
  }

  isCorrect() {
    let result = true;
    this.piece.forEach((piece) => {
      const x = (piece.id % this.divX);
      const y = Math.floor(piece.id / this.divX);
      if (piece.x !== x) result = false;
      if (piece.y !== y) result = false;
    });
    if (result === true) this.isEnd = true;
    return result;
  }

  updatePositions() {
    this.piece.forEach((piece) => {
      piece.sprite.x = this.x + (piece.x * piece.sprite.width);
      piece.sprite.y = this.y + (piece.y * piece.sprite.height);
    });
  }

  updateEnd() {
    //PLAY-SE
    AudioManager.playStaticSe({
      name: this.se.end,
      pitch: 100,
      volume: 100
    });
    this.endWait--;
    if (this.endWait > 0) {
      //const piece = this.piece.find(q => Number(q.sprite.alpha) < 1);
      //piece.sprite.alpha = (50 - this.endWait) / 50;
    } else {
      let result = true;
      this.items.forEach((item) => {
        SceneManager._scene.removeChild(item.sprite);
        if (item.isBat === false && item.isDiscarded === true) result = false;
        if (item.isBat === true && item.isDiscarded === false) result = false;
      });
      SceneManager._scene.removeChild(this.button.sprite);
      SceneManager._scene.removeChild(this.cursor.sprite);
      $gameSwitches.setValue(this.switchNo, result);
      // 判定
      return false;
    }
    return true;
  }

  escape() {
    this.piece.forEach((piece) => {
      SceneManager._scene.removeChild(piece.sprite);
    });
    $gameSwitches.setValue(this.switchNo, false);
    return false;
  }

  movePieceX(x) {
    let result = false;
    const tg = this.piece.find(q => q.sprite.alpha === 0);
    if (tg.x === 0 && x === 1) return;
    if (tg.x === (this.divX - 1) && x === -1) return;
    const list = this.piece.filter(q => q.y === tg.y)
      .sort((a, b) => { return a.x - b.x; });
    list.forEach((piece, i) => {
      if (!list[i + x]) return;
      const pos = [tg.x, piece.x];
      if (x > 0 && list[i + x].id === tg.id) {
        piece.x = pos[0];
        tg.x = pos[1];
      } else if (x < 0 && list[i + x].id === tg.id) {
        piece.x = pos[0];
        tg.x = pos[1];
      }
      result = true;
    });
    this.updatePositions();
    return result;
  }

  movePieceY(y) {
    let result = false;
    const tg = this.piece.find(q => q.sprite.alpha === 0);
    if (tg.y === 0 && y === 1) return;
    if (tg.y === (this.divY - 1) && y === -1) return;
    const list = this.piece.filter(q => q.x === tg.x)
      .sort((a, b) => { return a.y - b.y; });
    list.forEach((piece, i) => {
      if (!list[i + y]) return;
      const pos = [tg.y, piece.y];
      if (y > 0 && list[i + y].id === tg.id) {
        piece.y = pos[0];
        tg.y = pos[1];
      } else if (y < 0 && list[i + y].id === tg.id) {
        piece.y = pos[0];
        tg.y = pos[1];
      }
      result = true;
    });
    this.updatePositions();
    return result;
  }
}

//(function () {
//  "use strict";
//  var _TouchInput__onMouseMove = TouchInput._onMouseMove;
//  TouchInput._onMouseMove = function (event) {
//    _TouchInput__onMouseMove.apply(this, arguments);
//    this.mouseX = Graphics.pageToCanvasX(event.pageX);
//    this.mouseY = Graphics.pageToCanvasY(event.pageY);
//  };
//})();