/*:
 * @plugindesc モンプチ・スライドパズル
 * @author 夕多丸
 * 
 * @help
 * ■ プラグインコマンド
 * 
 * -----------------------------------------------------
 * PUZZLE switch file size divX divY x y missing
 * -----------------------------------------------------
 * switch  : 判定スイッチ番号 
 * file    : ファイル名 
 * size    : 画像サイズ 
 * divX    : 分割数X 
 * divY    : 分割数Y 
 * x       : 表示位置X 
 * y       : 表示位置Y 
 * missing : 欠損ピース 
 * =====================================================
 * 
 * -----------------------------------------------------
 * PUZZLE_SET option param
 * -----------------------------------------------------
 * 鍋の基本設定を用意
 * =====================================================
 * option  : 設定オプション名
 * param   : パラメータ
 * 
 * PUZZLE_SET correct [1,0,0,1,0,0]
 * 正解判定を行うピース（左上から右に順番で右下まで）
 * 
 * PUZZLE_SET pictures [1,2,3,4,5,6]
 * ピースに使用する画像番号（左上から右に順番で右下まで）
 * 
 * PUZZLE_SET shuffle 15 s20
 * シャッフルのパターン、もしくは混ぜる回数を指定
 * 
 * -----------------------------------------------------
 * PUZZLE_START
 * -----------------------------------------------------
 * 鍋のミニゲームを開始
 * 
/*:ja
 * @param se_move1
 * @desc 移動した時の音
 * @default Cursor1
 * @require 1
 * @dir audio/se/
 * @type file
 * 
 * @param se_move2
 * @desc 完了時のSE
 * @default Cursor2
 * @require 1
 * @dir audio/se/
 * @type file
 *
 * @param se_end
 * @desc 完了時のSE
 * @default Cursor1
 * @require 1
 * @dir audio/se/
 * @type file
 *
 **/
(function () {
  const parameters = PluginManager.parameters('MonpuchiPuzzle');
  const se = {
    move1: (parameters['se_move1'] || 'Cursor1'),
    move2: (parameters['se_move2'] || 'Cursor1'),
    end: (parameters['se_end'] || 'Cursor1'),
  };

  const _pluginCommand = Game_Interpreter.prototype.pluginCommand;
  Game_Interpreter.prototype.pluginCommand = function (command, args) {
    _pluginCommand.call(this, command, args);
    if (command.toUpperCase() === 'PUZZLE') {
      this.puzzle = new Puzzle(args);
      this.puzzle.se = se;
    }
    if (command.toUpperCase() === 'PUZZLE_SET') {
      this.puzzle.setOption(args);
    }
    if (command.toUpperCase() === 'PUZZLE_START') {
      this.setWaitMode("puzzle");
      this.puzzle.init();
    }
  };

  const _updateWaitMode = Game_Interpreter.prototype.updateWaitMode;
  Game_Interpreter.prototype.updateWaitMode = function () {
    var waiting = false;
    if (this._waitMode === "puzzle") {
      const active = this.puzzle.update();
      if (active === false) {
        this.puzzle = null;
        this._waitMode = "";
      }
      return active;
    }
    return _updateWaitMode.call(this);
  };
})();

class Puzzle {
  // PUZZLE 6 puzzle 150 3 3 50 30 8
  // switch, file, size, divX, divY, x, y, missing, pattern
  constructor(args) {
    this.switchNo = args[0];
    this.file = args[1];
    this.fileNumbers = [];
    this.size = Number(args[2]);
    this.divX = Number(args[3]);
    this.divY = Number(args[4]);
    this.x = Number(args[5]);
    this.y = Number(args[6]);
    this.missing = Number(args[7]);
    //
    this.piece = [];
    this.isMoving = false;
    this.isEnd = false;
    this.endWait = 50;
    this.skipFrame = 0;
    this.isPressed = false;
    this.isAnimate = false;
    //
    this.log = [];
    //
    this.pattern = "s50";
    this.shuffleWait = 15;
    if (this.pattern.substr(0, 1) === "s") {
      this.shuffleLength = Number(this.pattern.substr(1, this.pattern.length - 1));
      this.pattern = undefined;
    } else {
      this.shuffleLength = (!this.pattern) ? 100 : this.pattern.length;
    }
    //
    // 正解パターンを登録
    this.correctItems = [];
    for (let i = 0; i < this.x * this.y; i++) {
      this.correctItems.push(1);
      this.fileNumbers.push(i + 1);
    }
    this.shuffleLast = null;
    this.startWait = 50 + this.shuffleLength;
    //
    console.log(this);
  }

  init() {
    this.createImages();
  }

  setOption(args) {
    switch (args[0]) {
      case "correct":
        this.correctItems = JSON.parse(args[1]);
        break;
      case "pictures":
        this.fileNumbers = JSON.parse(args[1]);
        break;
      case "shuffle":
        this.shuffleWait = args[1];
        this.pattern = args[2];
        if (this.pattern.substr(0, 1) === "s") {
          this.shuffleLength = Number(this.pattern.substr(1, this.pattern.length - 1));
          this.pattern = undefined;
        } else {
          this.shuffleLength = (!this.pattern) ? 100 : this.pattern.length;
        }
        break;
    }
  }

  update() {
    if (this.isAnimate === true) return this.updateAnimations();
    if (this.isEnd === true) return this.updateEnd();
    if (this.startWait > this.shuffleLength) {
      this.startWait--;
      return true;
    }
    if (this.startWait > 0) {
      this.skipFrame++;
      if (this.skipFrame > this.shuffleWait) {
        this.skipFrame = 0;
        if (this.shuffle()) this.startWait--;
      }
    } else {
      if (Input.isTriggered('down')) this.movePieceY(1, true);
      if (Input.isTriggered('up')) this.movePieceY(-1, true);
      if (Input.isTriggered('left')) this.movePieceX(-1, true);
      if (Input.isTriggered('right')) this.movePieceX(1, true);
      if (Input.isTriggered('cancel')) return this.escape();
      this.isCorrect();
      if (TouchInput.isPressed() === true) {
        if (this.isPressed === false) this.moveClick();
      } else {
        this.isPressed = false;
      }
    }
    return true;
  }

  createImages() {
    const count = this.divX * this.divY;
    for (let i = 0; i < count; i++) {
      const filename = this.file + '/item' + (this.fileNumbers[i]);
      const bitmap = ImageManager.loadPicture(filename);
      const sprite = new Sprite(bitmap);
      const piece = {
        sprite: sprite,
        id: i,
        x: (i % this.divX),
        y: Math.floor(i / this.divX),
      };
      sprite.alpha = (i === this.missing) ? 0 : 1;
      this.piece.push(piece);
      SceneManager._scene.addChild(sprite);
    }
    this.updatePositions();
  }

  shuffle() {
    if (this.pattern !== undefined) return this.playPattern();
    let result = false;
    const sel = Math.floor(Math.random() * 4);
    if (sel === 0) result = this.movePieceY(1, false);
    if (sel === 1) result = this.movePieceY(-1, false);
    if (sel === 2) result = this.movePieceX(-1, false);
    if (sel === 3) result = this.movePieceX(1, false);
    if (result === true) {
      if (this.shuffleLast === 0 && sel === 1) result = false;
      if (this.shuffleLast === 1 && sel === 0) result = false;
      if (this.shuffleLast === 2 && sel === 3) result = false;
      if (this.shuffleLast === 3 && sel === 2) result = false;
      if (result === true) this.log = this.log + String(sel);
    }
    return result;
  }

  playPattern() {
    let result = false;
    if (this.pattern.length === 0) return;
    const sel = Number(this.pattern.substr(0, 1));
    this.pattern = this.pattern.substr(1, this.pattern.length - 1);
    if (sel === 0) result = this.movePieceY(1, false);
    if (sel === 1) result = this.movePieceY(-1, false);
    if (sel === 2) result = this.movePieceX(-1, false);
    if (sel === 3) result = this.movePieceX(1, false);
    return result;
  }

  isCorrect() {
    let result = true;
    this.piece.forEach((piece, i) => {
      if (this.correctItems[i] === 0) return;
      const x = (piece.id % this.divX);
      const y = Math.floor(piece.id / this.divX);
      if (piece.x !== x) result = false;
      if (piece.y !== y) result = false;
    });
    if (result === true) this.isEnd = true;
    return result;
  }

  updatePositions() {
    this.piece.forEach((piece) => {
      piece.sprite.x = this.x + (piece.x * this.size);
      piece.sprite.y = this.y + (piece.y * this.size);
    });
  }

  updateAnimations() {
    const move = this.size / 10;
    let hit = false;
    this.piece.forEach((piece) => {
      const vx = this.x + (piece.x * this.size);
      const vy = this.y + (piece.y * this.size);
      if (vx !== piece.sprite.x) {
        if (vx > piece.sprite.x) {
          piece.sprite.x += move;
        } else {
          piece.sprite.x -= move;
        }
        hit = true;
      }
      if (vy !== piece.sprite.y) {
        if (vy > piece.sprite.y) {
          piece.sprite.y += move;
        } else {
          piece.sprite.y -= move;
        }
        hit = true;
      }
    });
    if (hit === false) this.isAnimate = false;
    return true;
  }

  updateEnd() {
    this.endWait--;
    if (this.endWait === 0) AudioManager.playStaticSe({ //PLAY-SE
      name: this.se.end,
      pitch: 100,
      volume: 100
    });
    if (this.endWait > 0) {
      this.piece.forEach((piece, i) => {
        if (this.correctItems[i] === 1) {
          if (piece.sprite.alpha !== 1) piece.sprite.alpha = (50 - this.endWait) / 50;
        } else {
          if (piece.sprite.alpha !== 0) piece.sprite.alpha = (this.endWait) / 50;
        }
      });
    } else {
      this.piece.forEach((piece) => {
        SceneManager._scene.removeChild(piece.sprite);
      });
      if (this.endWait < -50) {
        $gameSwitches.setValue(this.switchNo, true);
        return false;
      }
    }
    return true;
  }

  escape() {
    this.piece.forEach((piece) => {
      SceneManager._scene.removeChild(piece.sprite);
    });
    $gameSwitches.setValue(this.switchNo, false);
    return false;
  }

  moveClick() {
    this.isPressed = true;
    //
    const vx = TouchInput.mouseX;
    const vy = TouchInput.mouseY;
    const areaCheck = (spr) => {
      if (vx < Number(spr.x)) return false;
      if (vy < Number(spr.y)) return false;
      if (vx > Number(spr.x) + Number(spr.width)) return false;
      if (vy > Number(spr.y) + Number(spr.height)) return false;
      return true;
    }
    const pieceCheck = (ih, ix, iy, button) => {
      if (ih === true) return;
      const vp = this.piece.find((p) => p.x === ix && p.y === iy);
      if (vp) {
        if (vp.sprite.alpha === 0) {
          ih = true;
          if (button === 'down') this.movePieceY(-1, true);
          if (button === 'up') this.movePieceY(1, true);
          if (button === 'left') this.movePieceX(1, true);
          if (button === 'right') this.movePieceX(-1, true);
        }
      }
      return ih;
    }
    let hit = false;
    this.piece.forEach((item) => {
      if (hit === true) return;
      if (areaCheck(item.sprite) === true) {
        console.log(item);
        hit = pieceCheck(hit, item.x - 1, item.y, 'right');
        hit = pieceCheck(hit, item.x + 1, item.y, 'left');
        hit = pieceCheck(hit, item.x, item.y - 1, 'down');
        hit = pieceCheck(hit, item.x, item.y + 1, 'up');
      }
    });
  }

  movePieceX(x, se) {
    let result = false;
    const tg = this.piece.find(q => q.sprite.alpha === 0);
    if (tg.x === 0 && x === 1) return this.playSe(false, se);
    if (tg.x === (this.divX - 1) && x === -1) return this.playSe(false, se);
    const list = this.piece.filter(q => q.y === tg.y)
      .sort((a, b) => { return a.x - b.x; });
    list.forEach((piece, i) => {
      if (!list[i + x]) return;
      const pos = [tg.x, piece.x];
      if (x > 0 && list[i + x].id === tg.id) {
        piece.x = pos[0];
        tg.x = pos[1];
      } else if (x < 0 && list[i + x].id === tg.id) {
        piece.x = pos[0];
        tg.x = pos[1];
      }
      result = true;
    });
    this.playSe(result, se);
    if (se === true) {
      this.isAnimate = true;
    } else {
      this.updatePositions();
    }
    return result;
  }

  movePieceY(y, se) {
    let result = false;
    const tg = this.piece.find(q => q.sprite.alpha === 0);
    if (tg.y === 0 && y === 1) return this.playSe(false, se);
    if (tg.y === (this.divY - 1) && y === -1) return this.playSe(false, se);
    const list = this.piece.filter(q => q.x === tg.x)
      .sort((a, b) => { return a.y - b.y; });
    list.forEach((piece, i) => {
      if (!list[i + y]) return;
      const pos = [tg.y, piece.y];
      if (y > 0 && list[i + y].id === tg.id) {
        piece.y = pos[0];
        tg.y = pos[1];
      } else if (y < 0 && list[i + y].id === tg.id) {
        piece.y = pos[0];
        tg.y = pos[1];
      }
      result = true;
    });
    this.playSe(result, se);
    if (se === true) {
      this.isAnimate = true;
    } else {
      this.updatePositions();
    }
    return result;
  }

  playSe(result, play) {
    if (play === false) return;
    AudioManager.playStaticSe({ //PLAY-SE
      name: (result === true) ? this.se.move1 : this.se.move2,
      pitch: 100,
      volume: 100
    });
  }
}
