/*:
 * @plugindesc モンプチ・トレーディングカードゲーム
 * @author 夕多丸
 * 
 * @help
 * ■ プラグインコマンド
 * 
 * -----------------------------------------------------
 * TCG switch file size divX divY x y missing
 * -----------------------------------------------------
 * switch  : 判定スイッチ番号 
 * file    : ファイル名 
 * size    : 画像サイズ 
 * divX    : 分割数X 
 * divY    : 分割数Y 
 * x       : 表示位置X 
 * y       : 表示位置Y 
 * missing : 欠損ピース 
 * =====================================================
 * 
 * -----------------------------------------------------
 * TCG_START
 * -----------------------------------------------------
 * TCGのミニゲームを開始
 * 
/*:ja
 *
 **/
(function () {
  // const parameters = PluginManager.parameters('MonpuchiTcg');
  //
  // プラグインコマンド設定
  const _pluginCommand = Game_Interpreter.prototype.pluginCommand;
  Game_Interpreter.prototype.pluginCommand = function (command, args) {
    _pluginCommand.call(this, command, args);
    if (command.toUpperCase() === 'TCG_INIT') {
      this.tcg = new Tcg(args);
    }
    if (command.toUpperCase() === 'TCG_SET') {
      this.tcg.setOption(args);
    }
    if (command.toUpperCase() === 'TCG_START') {
      this.setWaitMode("tcg");
      this.tcg.init();
    }
  };

  // メイン割り込み処理
  const _updateWaitMode = Game_Interpreter.prototype.updateWaitMode;
  Game_Interpreter.prototype.updateWaitMode = function () {
    var waiting = false;
    if (this._waitMode === "tcg") {
      const active = this.tcg.update();
      // if (active === false) {
      //   this.tcg = null;
      //   this._waitMode = "";
      // }
      return active;
    }
    return _updateWaitMode.call(this);
  };
})();

class Tcg {
  constructor(args) {
    this.cardList = this.getCardSetting();
    this.data = null;
    this.step = 0;
    //
    this.maxCardLength = 4;
    //
    this.key = {
      ok: false,
      cancel: false,
      up: false,
      down: false,
      left: false,
      right: false
    };
    //
    this.animations = [];
  }

  init() {
    console.log(this);
    //
    this.bg = this.createBg();
    this.step = 0;
    this.data = {
      p: this.initActor(this.getPlayer()),
      e: this.initActor(this.getEnemy())
    };
    this.selectCard = this.createSelect("card");
    this.selectArea = this.createSelect("area");
    //
    return;
  }

  update() {
    this.updateStock();
    this.updateStatus();
    //
    // アニメーション中
    if (this.updateAnimations() === true) return true;
    //
    if (this.step === 0) return this.phasePreparation();
    if (this.step === 1) return this.phasePlTurn();
    if (this.step === 2) return this.phaseOpenCard();
    if (this.step === 3) return this.phasePlAction();
    if (this.step === 4) return this.phasePlCounter();
    if (this.step === 5) return this.phasePlResult();
    if (this.step === 6) return this.phaseEnTurn();
    if (this.step === 7) return this.phaseEnAction();
    if (this.step === 8) return this.phaseEnCounter();
    if (this.step === 9) return this.phaseEnResult();
    if (this.step === 10) this.step = 1;
    //
    // 効果発動
    return true;
  }

  // 準備（カードを配る）
  phasePreparation() {
    const phand = this.data.p.deck.filter(q => q.status === "hand");
    const ehand = this.data.e.deck.filter(q => q.status === "hand");
    if (ehand.length < 4) {
      if (phand.length === ehand.length) {
        this.distributeCard("p");
      } else {
        this.distributeCard("e");
      }
    } else {
      this.step++;
      this.updateCardActive();
    }
    return true;
  }

  phaseOpenCard() {
    this.data.p.deck.filter(q => q.status === "hand").forEach((vcard) => {
      if (vcard.side === "back") {
        vcard.animations.push({ count: 10, data: [{ p: "scale.x", val: 0 }] });
        vcard.animations.push({ side: "front" });
        vcard.animations.push({ count: 10, data: [{ p: "scale.x", val: 0.5 }] });
        vcard.side = "front";
      }
    });
    this.step++;
    return true;
  }

  updateCardActive() {
    const vmp = this.data.p.mp;
    this.data.p.deck.filter(q => q.status === "hand").forEach((card) => {
      if (vmp.value >= card.type.cost) {
        card.spriteF.alpha = 1;
      } else {
        card.spriteF.alpha = 0.5;
      }
    });
  }

  // プレイヤーターン
  phasePlTurn() {
    this.data.p.mp.value++;
    this.setSelectVisible(0);
    //
    // 配置カード初期化＆ドロー
    this.setCardActiveToDiscard();
    this.distributeCard("p");
    this.updateCardActive();
    this.step++;
    return true;
  }

  phasePlAction() {
    const hand = this.data.p.deck.filter(q => q.status === "hand");
    const getValue = (val, max) => {
      if (val < 0) return max - 1;
      if (val >= max) return 0;
      return val;
    }
    if (this.selectArea.value === null) {
      if (this.checkKey("left")) this.selectCard.value = getValue(this.selectCard.value - 1, hand.length);
      if (this.checkKey("right")) this.selectCard.value = getValue(this.selectCard.value + 1, hand.length);
      if (this.checkKey("ok")) this.setSelectCard();
      this.setSelectVisible(this.selectCard.value);
    } else {
      if (this.checkKey("left")) this.selectArea.value = getValue(this.selectArea.value - 1, 2);
      if (this.checkKey("right")) this.selectArea.value = getValue(this.selectArea.value + 1, 2);
      if (this.checkKey("cancel")) this.setSelectCard(false)
      if (this.checkKey("ok")) {
        if (this.selectArea.value === 0) {
          this.data.p.mp.value -= hand[this.selectCard.value].type.cost;
          this.setSelectArea(hand[this.selectCard.value], this.data.p.area, "action");
        } else {
          this.setSelectArea(hand[this.selectCard.value], this.data.p.area, "discard");
        }
        this.updateCardsLocation();
        this.step++;
      }
    }
    return true;
  }

  phasePlCounter() {
    this.step++;
    return true;
  }

  phasePlResult() {
    const vpc = this.data.p.deck.filter(q => q.status === "action");
    if (vpc.length > 0) {
      vpc[0].type.action(this.data.e, vpc[0]);
    }
    this.step++;
    return true;
  }

  // プレイヤーターン
  phaseEnTurn() {
    this.data.e.mp.value++;
    //
    // 配置カード初期化＆ドロー
    this.setCardActiveToDiscard();
    this.distributeCard("e");
    this.step++;
    return true;
  }

  phaseEnAction() {
    const hand = this.data.e.deck.filter(q => q.status === "hand");
    this.data.e.mp.value -= hand[0].type.cost;
    this.setSelectArea(hand[0], this.data.e.area, "action");
    //
    // カードを裏返す
    this.data.e.deck.filter(q => q.status === "action").forEach((vcard) => {
      if (vcard.side === "back") {
        vcard.animations.push({ count: 10, data: [{ p: "scale.x", val: 0 }] });
        vcard.animations.push({ side: "front" });
        vcard.animations.push({ count: 10, data: [{ p: "scale.x", val: 0.5 }] });
        vcard.side = "front";
      }
    });
    this.step++;
    return true;
  }

  phaseEnCounter() {
    this.step++;
    return true;
  }

  phaseEnResult() {
    const vec = this.data.e.deck.filter(q => q.status === "action");
    if (vec.length > 0) {
      vec[0].type.action(this.data.p, vec[0]);
    }
    this.step++;
    return true;
  }

  setCardActiveToDiscard() {
    ["p", "e"].forEach((side) => {
      this.data[side].deck.filter((q) => {
        if (q.status === "discard") return true;
        if (q.status === "action") return true;
        return false;
      }).forEach((card) => {
        card.status = "discard";
        card.animations.push({
          type: "ease_in_out", count: 30, data: [{ p: "alpha", val: 0 }]
        });
      });
    });
  }

  setSelectVisible(vstat = null) {
    if (vstat !== null) {
      const vtg = this.data.p.deck.filter(q => q.status === "hand")[vstat];
      this.selectCard.value = vstat;
      this.selectCard.alpha = 1;
      this.selectCard.sprite.x = vtg.x;
      this.selectCard.sprite.y = vtg.y;
      this.selectCard.sprite.alpha = 1;
    } else {
      this.selectCard.value = null;
      this.selectCard.alpha = 0;
      this.selectCard.sprite.alpha = 0;
    }
  }

  setSelectCard(vstat = true) {
    this.selectArea.value = (vstat === true) ? 0 : null;
    this.selectArea.alpha = (vstat === true) ? 1 : 0;
    if (vstat === false) {
      const vtg = this.data.p.area.action;
      this.selectArea.sprite.x = vtg.x;
      this.selectArea.sprite.y = vtg.y;
      this.selectArea.sprite.alpha = 0;
    } else {
      this.selectArea.sprite.alpha = 1;
    }
  }

  setSelectArea(fcard, farea, fstatus) {
    fcard.status = fstatus;
    fcard.animations.push({
      type: "ease_in_out",
      count: 30,
      data: [
        { p: "x", val: farea[fstatus].x },
        { p: "y", val: farea[fstatus].y },
        { p: "scale.x", val: (fstatus === "action") ? 0.5 : 0.4 },
        { p: "scale.y", val: (fstatus === "action") ? 0.5 : 0.4 }
      ]
    });
    this.setSelectCard(false);
    this.setSelectVisible(null);
  }

  getPlayer() {
    return {
      face: "face_01",
      status: [80, 276],
      action: [313, 224],
      discard: [602, 260],
      stock: [750, 260],
      hand: { y: 430 - 180 / 2 },
      deck: [1, 2, 3, 1, 2, 3, 1, 2, 3, 3, 1, 2, 3, 1, 2, 3, 1, 2, 3, 3, 1, 2, 3, 1, 2, 3, 1, 2, 3, 3]
    }
  }

  getEnemy() {
    return {
      face: "face_02",
      status: [536, 106],
      action: [343, 90],
      discard: [86, 90],
      stock: [-62, 90],
      hand: { y: -120 - 180 / 2 },
      deck: [1, 2, 3, 1, 2, 3, 1, 2, 3, 3, 1, 2, 3, 1, 2, 3, 1, 2, 3, 3, 1, 2, 3, 1, 2, 3, 1, 2, 3, 3]
    }
  }

  initActor(fd) {
    const vst = this.createStatus(fd.face, fd.status[0], fd.status[1]);
    vst.deck = [];
    fd.deck.forEach((vid, vi) => {
      vst.deck.push({
        num: vi,
        type: this.cardList.find(q => q.id === vid),
        status: "stock",
        animations: []
      });
    });
    vst.area = {
      action: this.createArea("area_action", fd.action[0], fd.action[1]),
      discard: this.createArea("area_discard", fd.discard[0], fd.discard[1]),
      stock: this.createStock("area_stock", fd.stock[0], fd.stock[1]),
      hand: fd.hand
    }
    return vst;
  }

  checkKey(code) {
    if (Input.isPressed(code) === true) {
      if (this.key[code] === false) {
        this.key[code] = true;
        return true;
      }
    } else {
      this.key[code] = false;
    }
    return false;
  }

  updateStatus() {
    const setFrame = (vstat) => {
      let vn1 = Math.floor(vstat.value / 10);
      let vn2 = vstat.value % 10;
      if (vn1 === 0) vn1 = 10;
      vstat.sprite1.setFrame(vn1 * 20, 0, 20, 30);
      vstat.sprite2.setFrame(vn2 * 20, 0, 20, 30);
    }
    setFrame(this.data.p.hp);
    setFrame(this.data.p.mp);
    setFrame(this.data.e.hp);
    setFrame(this.data.e.mp);
  }

  updateAnimations() {
    let result = false;
    //
    const vselA = this.selectArea;
    const varea = ["action", "discard"]; // "people"
    if (vselA.value !== null) {
      const vtg = this.data.p.area[varea[vselA.value]];
      vselA.sprite.x = vtg.x;
      vselA.sprite.y = vtg.y;
      vselA.sprite.scale.x = (vselA.value === 0) ? 1 : 0.8;
      vselA.sprite.scale.y = (vselA.value === 0) ? 1 : 0.8;
      vselA.sprite.alpha = 1;
    }
    if (this.updateAnimationsCard() === true) result = true;
    return result;
  }

  updateAnimationsCard() {
    let result = false;
    ["p", "e"].forEach((side) => {
      this.data[side].deck.forEach((card) => {
        if (card.animations.length === 0) return;
        const vani = card.animations[0];
        if (typeof vani.count === "number") vani.count = [0, vani.count];
        if (vani.type === undefined) vani.type = "ease_in_out";
        if (vani.side !== undefined) {
          card.spriteF.visible = (vani.side === "front") ? true : false;
          card.spriteB.visible = (vani.side === "front") ? false : true;
          card.side = vani.side;
          card.animations.shift();
        } else if (typeof vani.func === "function") {
          vani.func();
          card.animations.shift();
        } else if (vani.count[0] === vani.count[1]) {
          card.animations.shift();
        } else {
          vani.count[0]++;
          vani.data.forEach((vset) => {
            const vkey = vset.p.split(".");
            if (vkey.length === 1) {
              const vside = (card.side === "front") ? card.spriteF[vset.p] : card.spriteB[vset.p];
              if (vset.org === undefined) vset.org = vside;
              const val1 = this.getEase(vani.type, vani.count[0], vset.org, vset.val, vani.count[1]);
              card.spriteF[vset.p] = val1;
              card.spriteB[vset.p] = val1;
              if (vset.p === "x") card[vset.p] = val1;
              if (vset.p === "y") card[vset.p] = val1;
            } else {
              if (vset.org === undefined) vset.org = card.spriteF[vkey[0]][vkey[1]];
              const val2 = this.getEase(vani.type, vani.count[0], vset.org, vset.val, vani.count[1]);
              card.spriteF[vkey[0]][vkey[1]] = val2;
              card.spriteB[vkey[0]][vkey[1]] = val2;
            }
          });
          result = true;
        }
      });
    });
    return result;
  }

  updateStock() {
    ["p", "e"].forEach((side) => {
      const vnum = this.data[side].deck.filter(q => q.status === "stock").length;
      if (this.data[side].area.stock.value === vnum) return;
      let vn1 = Math.floor(vnum / 10);
      let vn2 = vnum % 10;
      if (vn1 === 0) vn1 = 10;
      this.data[side].area.stock.num1.setFrame(vn1 * 20, 0, 20, 30);
      this.data[side].area.stock.num2.setFrame(vn2 * 20, 0, 20, 30);
      this.data[side].area.stock.cardFront.y = this.data[side].area.stock.cardY - Math.floor(vnum / 2)
    });
  }

  distributeCard(fside) {
    const vhand = this.data[fside].deck.filter(q => q.status === "hand");
    if (vhand.length >= this.maxCardLength) return;
    const vstock = this.data[fside].deck.filter(q => q.status === "stock");
    const vcard = this.getCardSprite(vstock[0], this.data[fside]);
    vstock[0].status = "hand";
    vstock[0].spriteF = vcard.spriteF;
    vstock[0].spriteB = vcard.spriteB;
    vstock[0].side = "back";
    this.updateCardsLocation();
  }

  // カードの位置を更新
  updateCardsLocation() {
    const cx = 160 / 2;
    // プレイヤーの手札
    const vpc = this.data.p.deck.filter(q => q.status === "hand");
    const vpl = (816 - (vpc.length * 180 - 20)) / 2;
    vpc.forEach((vcard, vi) => {
      vcard.animations.push({
        count: 20, data: [
          { p: "x", val: vpl + vi * 180 + cx },
          { p: "y", val: this.data.p.area.hand.y + (360 / 2) },
          { p: "scale.x", val: 0.5 },
          { p: "scale.y", val: 0.5 }
        ]
      });
    });
    // 相手の手札
    const vec = this.data.e.deck.filter(q => q.status === "hand");
    const vel = (816 - (vec.length * 180 - 20)) / 2;
    vec.forEach((vcard, vi) => {
      vcard.animations.push({
        count: 20, data: [
          { p: "x", val: vel + (vec.length - 1 - vi) * 180 + cx },
          { p: "y", val: this.data.e.area.hand.y + (360 / 2) },
          { p: "scale.x", val: 0.5 },
          { p: "scale.y", val: 0.5 }
        ]
      });
    });
  }

  getCardSprite(fcard, fside) {
    const setDefault = (sprite, visible) => {
      sprite.visible = visible;
      sprite.alpha = 1;
      sprite.anchor.x = 0.5;
      sprite.anchor.y = 0.5;
      sprite.x = fside.area.stock.x;
      sprite.y = fside.area.stock.y;
      sprite.scale.x = 0.4;
      sprite.scale.y = 0.4;
      return sprite;
    }
    const bitmapF = ImageManager.loadPicture('/card/' + fcard.type.image);
    const bitmapB = ImageManager.loadPicture('/card/card_ura');
    const spriteF = setDefault(new Sprite(bitmapF), false);
    const spriteB = setDefault(new Sprite(bitmapB), true);
    SceneManager._scene.addChild(spriteF);
    SceneManager._scene.addChild(spriteB);
    return {
      spriteF: spriteF,
      spriteB: spriteB
    };
  }

  getAreaSprite(file, fx, fy, falpha = 1) {
    const filename = '/card/' + file;
    const bitmap = ImageManager.loadPicture(filename);
    const sprite = new Sprite(bitmap);
    sprite.alpha = falpha;
    sprite.anchor.x = 0.5;
    sprite.anchor.y = 0.5;
    sprite.x = fx;
    sprite.y = fy;
    SceneManager._scene.addChild(sprite);
    return sprite;
  }

  createArea(file, vx, vy) {
    const cx = 128 / 2;
    const cy = 144 / 2;
    return {
      x: vx + cx,
      y: vy + cy,
      sprite: this.getAreaSprite(file, vx + cx, vy + cy)
    };
  }

  createBg() {
    return this.getAreaSprite("background", 408, 312, 1);
  }

  createSelect(ftype) {
    return {
      value: null,
      sprite: this.getAreaSprite("select", 0, 0, 0)
    }
  }

  createStock(file, fx, fy) {
    const cx = 128 / 2;
    const cy = 144 / 2;
    const cs = (fx < 408) ? 28 : -28;
    return {
      x: fx + cx,
      y: fy + cy,
      value: 30,
      sprite: this.getAreaSprite(file, fx + cx, fy + cy),
      cardBack: this.getSprite("stock_back", 128, 20, 0, fx, fy + 124),
      cardFront: this.getSprite("stock_front", 128, 144, 0, fx, fy - 15),
      cardY: fy,
      num1: this.getSprite("number_2", 20, 30, 3, fx + cs + cx - 20, fy + cy + 15),
      num2: this.getSprite("number_2", 20, 30, 0, fx + cs + cx, fy + cy + 15),
    };
  }

  createStatus(fimg, fx, fy) {
    const face = this.getSprite(fimg, 90, 90, 0, fx + 0, fy + 10);
    face.anchor.x = 0.5;
    face.anchor.y = 0.5;
    face.x = fx + 45;
    face.y = fy + 45;
    //
    return {
      face: {
        sprite: face,
        x: face.x,
        y: face.y
      },
      name: this.getSprite("name_01", 160, 30, 0, fx + 110, fy + 0),
      hp: {
        value: 10,
        icon: this.getSprite("status_hp", 30, 30, 0, fx + 110, fy + 40),
        sprite1: this.getSprite("number_1", 20, 30, 1, fx + 160, fy + 40),
        sprite2: this.getSprite("number_1", 20, 30, 0, fx + 180, fy + 40),
      },
      mp: {
        value: 3,
        icon: this.getSprite("status_mp", 30, 30, 0, fx + 110, fy + 80),
        sprite1: this.getSprite("number_1", 20, 30, 10, fx + 160, fy + 80),
        sprite2: this.getSprite("number_1", 20, 30, 3, fx + 180, fy + 80)
      }
    }
  }

  playSe(result, play) {
    if (play === false) return;
    AudioManager.playStaticSe({ //PLAY-SE
      name: (result === true) ? this.se.move1 : this.se.move2,
      pitch: 100,
      volume: 100
    });
  }

  getSprite(vfile, fw, fh, fd, fx, fy) {
    const filename = '/card/' + vfile;
    const bitmap = ImageManager.loadPicture(filename);
    const sprite = new Sprite(bitmap);
    sprite.alpha = 1;
    sprite.x = fx;
    sprite.y = fy;
    sprite.setFrame((fd * fw), 0, fw, fh);
    SceneManager._scene.addChild(sprite);
    return sprite;
  }

  getCardSetting() {
    return [{
      id: 1,
      image: "card_001",
      cost: 1,
      action: (ft, fc,) => {
        console.log(fc, ft);
        fc.animations.push({
          count: 15,
          data: [
            { p: "x", val: ft.face.x },
            { p: "y", val: ft.face.y }
          ]
        });
        fc.animations.push({
          count: 5, data: [{ p: "y", val: ft.face.y + 30 }]
        });
        fc.animations.push({
          count: 5, data: [{ p: "y", val: ft.face.y }]
        });
        fc.animations.push({
          func: () => ft.hp.value--
        });
      }
    }, {
      id: 2,
      image: "card_002",
      cost: 3,
      action: (ft, fc,) => {
        console.log("002", ft);
        ft.hp.value--;
      }
    }, {
      id: 3,
      image: "card_003",
      cost: 1,
      action: (ft, fc,) => {
        console.log("003", ft);
        ft.hp.value--;
      }
    }];
  }

  getEase(type, t, from, to, d) {
    if (t === d) return to;
    const b = from;
    const c = to - from;
    //
    if (type === "ease_in") {
      t /= d;
      return c * t * t + b;
    }
    if (type === "ease_out") {
      t /= d;
      return -c * t * (t - 2.0) + b;
    }
    if (type === "ease_in_out") {
      t /= d / 2
      if (t < 1) return c / 2 * t * t + b
      t = t - 1
      return -c / 2 * (t * (t - 2) - 1) + b
    }
  }
}
